package Catalyst::Plugin::Shorten;
use 5.006; use strict; use warnings; our $VERSION = '0.01';

use Bijection qw/all/;

our ($pkey, $ukey, $short);

sub setup {
	my $c = shift;

	my $config = $c->config->{'Plugin::Shorten'};

	if ($config->{set}) {
		$c->shorten_bijection_set(@{$config->{set}});
	}

	$pkey = $config->{map}->{params} || 'params';
	$ukey = $config->{map}->{uri} || 'uri';
	$short = $config->{map}->{s} || 's';

	1;
}

sub shorten {
	my ($c, $asUrl, $additional) = @_;

	my $id = $c->shorten_set_data(
		$pkey => $c->req->params,
		$ukey => $c->req->uri,
		%{$additional}
	);

	my $string = biject($id);
	return $asUrl
		? $c->uri_for_action($c->action->private_path, { $short => $string })
		: $string;
}

sub shorten_delete {
	my ($c, $str) = @_;
	if (my $id = inverse($str)) {
		return $c->shorten_delete_data($id);
	}
}

sub shorten_extract {
	my ($c, $cb) = @_;

	my $params = $c->req->params;
	my $sparams = $c->shorten_params($params, $cb);

	if ($sparams) {
		delete $params->{$short};
		$sparams = {%{$sparams}, %{$params}};
		$c->req->parameters($sparams);
	}

	1;
}

sub shorten_params {
	my ($c, $params, $cb) = @_;
	$params ||= $c->req->params;
	if ($params->{$short}) {
		if (my $id = inverse($params->{$short})) {
			my $shorten = $c->shorten_get_data($id);
			if (defined $cb) {
				return $cb->($c, $shorten->{$pkey}, $shorten);
			}
			return $shorten->{$pkey};
		}
	}
}

sub shorten_redirect {
	my ($c, $str, $cb) = @_;
	if (my $id = inverse($str)) {
		my $shorten = $c->shorten_get_data($id);
		my $uri = $shorten->{$ukey};
		if (defined $cb) {
			$uri = $cb->($c, $shorten->{$ukey}, $shorten);
		}
		$c->res->redirect($uri) if $uri;
	}
}

sub shorten_bijection_set {
	my ($c, @set) = @_;
	bijection_set(@set);
}

1;

__END__

=head1 NAME

Catalyst::Plugin::Shorten - The great ancient URI shortner!

=head1 VERSION

Version 0.01

=cut

=head1 SYNOPSIS

	use Catalyst qw/
		Shorten
		Shorten::Store::Dummy
	/;

	sub auto :Path :Args(0) {
		my ($self, $c) = @_;
		$c->shorten_extract; # checks whether the shorten param exists if it does merges the stored params into the request
	}
	
	........

	sub endpoint :Chained('base') :PathPart('ending') :Args('0') {
		my ($self, $c) = @_;

		my $str = $c->shorten(); # returns bijection references to an ID in the store.
		my $url = $c->shorten(1); # return a url to the current endpoint replacing all params with localhost:300/ending?s=GH
	}
	
	-------

	use Catalyst qw/
		Shorten
		Shorten::Store::Dummy
	/;
	
	__PACKAGE__->config(
		......
		'Plugin::Shorten' => {
			set => [qw/c b a ..../],
			map => {
				params => 'data',
				uri => 'url',
			}
		}
	);

	package TestApp::Controller::Shorten;

	use Moose;
	use namespace::autoclean;

	BEGIN {
		extends 'Catalyst::Controller';
	}
		
	sub g :Chained('/') :PathPart('g') :Args('1') {
		my ($self, $c, $cap) = @_;
		$c->shorten_redirect($cap);
	}

	__PACKAGE__->meta->make_immutable;

	1;

=head1 SUBROUTINES/METHODS

=head2 shorten (asUrl, $additional)

Take the current request uri and store, returns an Bijective string.

=cut

=head2 shorten_delete (str)

Delete from storage.

=cut

=head2 shorten_extract (cb)

Check for the param (default is 's'), if defined attempt to inverse and then right merge with the current requests params.

This always returns true and you can later access the merged params using - 
	
	$c->req->params;

=cut

=head2 shorten_params (params, cb)

Check for the param (default is 's'), if defined attempt to inverse and then return the params retrieved from storage.

=cut

=head2 shorten_redirect (str, cb)

Redirect the clients browser to the uri retrieved from the storage.

=cut

=head1 AUTHOR

LNATION, C<< <thisusedtobeanemail at gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-catalyst-plugin-shorten at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Catalyst-Plugin-Shorten>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Catalyst::Plugin::Shorten


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Catalyst-Plugin-Shorten>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Catalyst-Plugin-Shorten>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Catalyst-Plugin-Shorten>

=item * Search CPAN

L<http://search.cpan.org/dist/Catalyst-Plugin-Shorten/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2018 LNATION.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Catalyst::Plugin::Shorten
