"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.findBootstrapApplicationCall = exports.addModuleImportToStandaloneBootstrap = exports.importsProvidersFrom = void 0;
const schematics_1 = require("@angular-devkit/schematics");
const change_1 = require("@schematics/angular/utility/change");
const ts = require("typescript");
const ast_utils_1 = require("@schematics/angular/utility/ast-utils");
/**
 * Checks whether the providers from a module are being imported in a `bootstrapApplication` call.
 * @param tree File tree of the project.
 * @param filePath Path of the file in which to check.
 * @param className Class name of the module to search for.
 */
function importsProvidersFrom(tree, filePath, className) {
    const sourceFile = ts.createSourceFile(filePath, tree.readText(filePath), ts.ScriptTarget.Latest, true);
    const bootstrapCall = findBootstrapApplicationCall(sourceFile);
    const importProvidersFromCall = bootstrapCall ? findImportProvidersFromCall(bootstrapCall) : null;
    return (!!importProvidersFromCall &&
        importProvidersFromCall.arguments.some(arg => ts.isIdentifier(arg) && arg.text === className));
}
exports.importsProvidersFrom = importsProvidersFrom;
/**
 * Adds an `importProvidersFrom` call to the `bootstrapApplication` call.
 * @param tree File tree of the project.
 * @param filePath Path to the file that should be updated.
 * @param moduleName Name of the module that should be imported.
 * @param modulePath Path from which to import the module.
 */
function addModuleImportToStandaloneBootstrap(tree, filePath, moduleName, modulePath) {
    const sourceFile = ts.createSourceFile(filePath, tree.readText(filePath), ts.ScriptTarget.Latest, true);
    const bootstrapCall = findBootstrapApplicationCall(sourceFile);
    if (!bootstrapCall) {
        throw new schematics_1.SchematicsException(`Could not find bootstrapApplication call in ${filePath}`);
    }
    const recorder = tree.beginUpdate(filePath);
    const importCall = findImportProvidersFromCall(bootstrapCall);
    const printer = ts.createPrinter();
    const sourceText = sourceFile.getText();
    // Add imports to the module being added and `importProvidersFrom`. We don't
    // have to worry about duplicates, because `insertImport` handles them.
    [
        (0, ast_utils_1.insertImport)(sourceFile, sourceText, moduleName, modulePath),
        (0, ast_utils_1.insertImport)(sourceFile, sourceText, 'importProvidersFrom', '@angular/core'),
    ].forEach(change => {
        if (change instanceof change_1.InsertChange) {
            recorder.insertLeft(change.pos, change.toAdd);
        }
    });
    // If there is an `importProvidersFrom` call already, reuse it.
    if (importCall) {
        recorder.insertRight(importCall.arguments[importCall.arguments.length - 1].getEnd(), `, ${moduleName}`);
    }
    else if (bootstrapCall.arguments.length === 1) {
        // Otherwise if there is no options parameter to `bootstrapApplication`,
        // create an object literal with a `providers` array and the import.
        const newCall = ts.factory.updateCallExpression(bootstrapCall, bootstrapCall.expression, bootstrapCall.typeArguments, [
            ...bootstrapCall.arguments,
            ts.factory.createObjectLiteralExpression([createProvidersAssignment(moduleName)], true),
        ]);
        recorder.remove(bootstrapCall.getStart(), bootstrapCall.getWidth());
        recorder.insertRight(bootstrapCall.getStart(), printer.printNode(ts.EmitHint.Unspecified, newCall, sourceFile));
    }
    else {
        const providersLiteral = findProvidersLiteral(bootstrapCall);
        if (providersLiteral) {
            // If there's a `providers` array, add the import to it.
            const newProvidersLiteral = ts.factory.updateArrayLiteralExpression(providersLiteral, [
                ...providersLiteral.elements,
                createImportProvidersFromCall(moduleName),
            ]);
            recorder.remove(providersLiteral.getStart(), providersLiteral.getWidth());
            recorder.insertRight(providersLiteral.getStart(), printer.printNode(ts.EmitHint.Unspecified, newProvidersLiteral, sourceFile));
        }
        else {
            // Otherwise add a `providers` array to the existing object literal.
            const optionsLiteral = bootstrapCall.arguments[1];
            const newOptionsLiteral = ts.factory.updateObjectLiteralExpression(optionsLiteral, [
                ...optionsLiteral.properties,
                createProvidersAssignment(moduleName),
            ]);
            recorder.remove(optionsLiteral.getStart(), optionsLiteral.getWidth());
            recorder.insertRight(optionsLiteral.getStart(), printer.printNode(ts.EmitHint.Unspecified, newOptionsLiteral, sourceFile));
        }
    }
    tree.commitUpdate(recorder);
}
exports.addModuleImportToStandaloneBootstrap = addModuleImportToStandaloneBootstrap;
/** Finds the call to `bootstrapApplication` within a file. */
function findBootstrapApplicationCall(sourceFile) {
    const localName = findImportLocalName(sourceFile, 'bootstrapApplication', '@angular/platform-browser');
    return localName ? findCall(sourceFile, localName) : null;
}
exports.findBootstrapApplicationCall = findBootstrapApplicationCall;
/** Find a call to `importProvidersFrom` within a `bootstrapApplication` call. */
function findImportProvidersFromCall(bootstrapCall) {
    const providersLiteral = findProvidersLiteral(bootstrapCall);
    const importProvidersName = findImportLocalName(bootstrapCall.getSourceFile(), 'importProvidersFrom', '@angular/core');
    if (providersLiteral && importProvidersName) {
        for (const element of providersLiteral.elements) {
            // Look for an array element that calls the `importProvidersFrom` function.
            if (ts.isCallExpression(element) &&
                ts.isIdentifier(element.expression) &&
                element.expression.text === importProvidersName) {
                return element;
            }
        }
    }
    return null;
}
/** Finds the `providers` array literal within a `bootstrapApplication` call. */
function findProvidersLiteral(bootstrapCall) {
    // The imports have to be in the second argument of
    // the function which has to be an object literal.
    if (bootstrapCall.arguments.length > 1 &&
        ts.isObjectLiteralExpression(bootstrapCall.arguments[1])) {
        for (const prop of bootstrapCall.arguments[1].properties) {
            if (ts.isPropertyAssignment(prop) &&
                ts.isIdentifier(prop.name) &&
                prop.name.text === 'providers' &&
                ts.isArrayLiteralExpression(prop.initializer)) {
                return prop.initializer;
            }
        }
    }
    return null;
}
/**
 * Finds the local name of an imported symbol. Could be the symbol name itself or its alias.
 * @param sourceFile File within which to search for the import.
 * @param name Actual name of the import, not its local alias.
 * @param moduleName Name of the module from which the symbol is imported.
 */
function findImportLocalName(sourceFile, name, moduleName) {
    for (const node of sourceFile.statements) {
        // Only look for top-level imports.
        if (!ts.isImportDeclaration(node) ||
            !ts.isStringLiteral(node.moduleSpecifier) ||
            node.moduleSpecifier.text !== moduleName) {
            continue;
        }
        // Filter out imports that don't have the right shape.
        if (!node.importClause ||
            !node.importClause.namedBindings ||
            !ts.isNamedImports(node.importClause.namedBindings)) {
            continue;
        }
        // Look through the elements of the declaration for the specific import.
        for (const element of node.importClause.namedBindings.elements) {
            if ((element.propertyName || element.name).text === name) {
                // The local name is always in `name`.
                return element.name.text;
            }
        }
    }
    return null;
}
/**
 * Finds a call to a function with a specific name.
 * @param rootNode Node from which to start searching.
 * @param name Name of the function to search for.
 */
function findCall(rootNode, name) {
    let result = null;
    rootNode.forEachChild(function walk(node) {
        if (ts.isCallExpression(node) &&
            ts.isIdentifier(node.expression) &&
            node.expression.text === name) {
            result = node;
        }
        if (!result) {
            node.forEachChild(walk);
        }
    });
    return result;
}
/** Creates an `importProvidersFrom({{moduleName}})` call. */
function createImportProvidersFromCall(moduleName) {
    return ts.factory.createCallChain(ts.factory.createIdentifier('importProvidersFrom'), undefined, undefined, [ts.factory.createIdentifier(moduleName)]);
}
/** Creates a `providers: [importProvidersFrom({{moduleName}})]` property assignment. */
function createProvidersAssignment(moduleName) {
    return ts.factory.createPropertyAssignment('providers', ts.factory.createArrayLiteralExpression([createImportProvidersFromCall(moduleName)]));
}
//# sourceMappingURL=data:application/json;base64,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