﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/accessanalyzer/AccessAnalyzerRequest.h>
#include <aws/accessanalyzer/AccessAnalyzer_EXPORTS.h>
#include <aws/accessanalyzer/model/Configuration.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace AccessAnalyzer {
namespace Model {

/**
 */
class CreateAccessPreviewRequest : public AccessAnalyzerRequest {
 public:
  AWS_ACCESSANALYZER_API CreateAccessPreviewRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAccessPreview"; }

  AWS_ACCESSANALYZER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources">ARN
   * of the account analyzer</a> used to generate the access preview. You can only
   * create an access preview for analyzers with an <code>Account</code> type and
   * <code>Active</code> status.</p>
   */
  inline const Aws::String& GetAnalyzerArn() const { return m_analyzerArn; }
  inline bool AnalyzerArnHasBeenSet() const { return m_analyzerArnHasBeenSet; }
  template <typename AnalyzerArnT = Aws::String>
  void SetAnalyzerArn(AnalyzerArnT&& value) {
    m_analyzerArnHasBeenSet = true;
    m_analyzerArn = std::forward<AnalyzerArnT>(value);
  }
  template <typename AnalyzerArnT = Aws::String>
  CreateAccessPreviewRequest& WithAnalyzerArn(AnalyzerArnT&& value) {
    SetAnalyzerArn(std::forward<AnalyzerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Access control configuration for your resource that is used to generate the
   * access preview. The access preview includes findings for external access allowed
   * to the resource with the proposed access control configuration. The
   * configuration must contain exactly one element.</p>
   */
  inline const Aws::Map<Aws::String, Configuration>& GetConfigurations() const { return m_configurations; }
  inline bool ConfigurationsHasBeenSet() const { return m_configurationsHasBeenSet; }
  template <typename ConfigurationsT = Aws::Map<Aws::String, Configuration>>
  void SetConfigurations(ConfigurationsT&& value) {
    m_configurationsHasBeenSet = true;
    m_configurations = std::forward<ConfigurationsT>(value);
  }
  template <typename ConfigurationsT = Aws::Map<Aws::String, Configuration>>
  CreateAccessPreviewRequest& WithConfigurations(ConfigurationsT&& value) {
    SetConfigurations(std::forward<ConfigurationsT>(value));
    return *this;
  }
  template <typename ConfigurationsKeyT = Aws::String, typename ConfigurationsValueT = Configuration>
  CreateAccessPreviewRequest& AddConfigurations(ConfigurationsKeyT&& key, ConfigurationsValueT&& value) {
    m_configurationsHasBeenSet = true;
    m_configurations.emplace(std::forward<ConfigurationsKeyT>(key), std::forward<ConfigurationsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A client token.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateAccessPreviewRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_analyzerArn;
  bool m_analyzerArnHasBeenSet = false;

  Aws::Map<Aws::String, Configuration> m_configurations;
  bool m_configurationsHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace AccessAnalyzer
}  // namespace Aws
