﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/b2bi/B2BIRequest.h>
#include <aws/b2bi/B2BI_EXPORTS.h>
#include <aws/b2bi/model/Logging.h>
#include <aws/b2bi/model/Tag.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace B2BI {
namespace Model {

/**
 */
class CreateProfileRequest : public B2BIRequest {
 public:
  AWS_B2BI_API CreateProfileRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateProfile"; }

  AWS_B2BI_API Aws::String SerializePayload() const override;

  AWS_B2BI_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the name of the profile.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateProfileRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the email address associated with this customer profile.</p>
   */
  inline const Aws::String& GetEmail() const { return m_email; }
  inline bool EmailHasBeenSet() const { return m_emailHasBeenSet; }
  template <typename EmailT = Aws::String>
  void SetEmail(EmailT&& value) {
    m_emailHasBeenSet = true;
    m_email = std::forward<EmailT>(value);
  }
  template <typename EmailT = Aws::String>
  CreateProfileRequest& WithEmail(EmailT&& value) {
    SetEmail(std::forward<EmailT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the phone number associated with the profile.</p>
   */
  inline const Aws::String& GetPhone() const { return m_phone; }
  inline bool PhoneHasBeenSet() const { return m_phoneHasBeenSet; }
  template <typename PhoneT = Aws::String>
  void SetPhone(PhoneT&& value) {
    m_phoneHasBeenSet = true;
    m_phone = std::forward<PhoneT>(value);
  }
  template <typename PhoneT = Aws::String>
  CreateProfileRequest& WithPhone(PhoneT&& value) {
    SetPhone(std::forward<PhoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the name for the business associated with this profile.</p>
   */
  inline const Aws::String& GetBusinessName() const { return m_businessName; }
  inline bool BusinessNameHasBeenSet() const { return m_businessNameHasBeenSet; }
  template <typename BusinessNameT = Aws::String>
  void SetBusinessName(BusinessNameT&& value) {
    m_businessNameHasBeenSet = true;
    m_businessName = std::forward<BusinessNameT>(value);
  }
  template <typename BusinessNameT = Aws::String>
  CreateProfileRequest& WithBusinessName(BusinessNameT&& value) {
    SetBusinessName(std::forward<BusinessNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether or not logging is enabled for this profile.</p>
   */
  inline Logging GetLogging() const { return m_logging; }
  inline bool LoggingHasBeenSet() const { return m_loggingHasBeenSet; }
  inline void SetLogging(Logging value) {
    m_loggingHasBeenSet = true;
    m_logging = value;
  }
  inline CreateProfileRequest& WithLogging(Logging value) {
    SetLogging(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reserved for future use.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateProfileRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the key-value pairs assigned to ARNs that you can use to group and
   * search for resources by type. You can attach this metadata to resources
   * (capabilities, partnerships, and so on) for any purpose.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateProfileRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateProfileRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_email;
  bool m_emailHasBeenSet = false;

  Aws::String m_phone;
  bool m_phoneHasBeenSet = false;

  Aws::String m_businessName;
  bool m_businessNameHasBeenSet = false;

  Logging m_logging{Logging::NOT_SET};
  bool m_loggingHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace B2BI
}  // namespace Aws
