#include "simdutf.h"

#include <array>
#include <vector>

#include <tests/helpers/fixed_string.h>
#include <tests/helpers/random_int.h>
#include <tests/helpers/test.h>
#include <tests/helpers/transcode_test_base.h>

namespace {
constexpr std::array<size_t, 7> input_size{7, 16, 12, 64, 67, 128, 256};
constexpr simdutf::endianness BE = simdutf::endianness::BIG;

using simdutf::tests::helpers::transcode_utf16_to_utf32_test_base;

} // namespace

TEST(issue_convert_utf16be_to_utf32_with_errors_7f6091b746e6e764) {
  alignas(2) const unsigned char data[] = {
      0xc2, 0x90, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x01, 0x00, 0x00, 0x03, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x05,
      0x00, 0x23, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x23, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x05, 0x00,
      0x5e, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53,
      0x53, 0x53, 0x53, 0x53, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0xf2, 0xa8,
      0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00, 0x00, 0x00, 0x5e, 0x5e, 0x5e, 0x5e,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e,
      0x5e, 0x5e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 0x53, 0x53,
      0x53, 0x0b, 0x89, 0x0a, 0x00, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53,
      0x53, 0x53, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0xf2, 0xa8, 0xa4, 0x8b,
      0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x53,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00,
      0x53, 0x00, 0x59, 0x00, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5b, 0x23,
      0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x05, 0x00, 0x40, 0xf2, 0xa8, 0xa4,
      0x8b, 0x7a, 0x01, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x57, 0x53, 0x53, 0x53, 0x53, 0x12, 0x0b, 0x41,
      0x12, 0x41, 0x0b, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53,
      0x53, 0x53, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0xf2, 0xa8, 0xa4, 0x8b,
      0x01, 0x5b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x53, 0x53, 0x53, 0x53, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x74,
      0xcc, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e,
      0x5e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0xa8,
      0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x53, 0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5b, 0x23, 0x00,
      0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x05, 0x00, 0x23, 0xf2, 0xa8, 0xa4, 0x8b,
      0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00, 0x5e,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x23, 0x5e, 0x5e, 0x5e, 0x5e,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0xa8, 0xa4,
      0x8b, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00, 0x53, 0x53, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x5b, 0x23, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01,
      0x05, 0x00, 0x23, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x23, 0xf2, 0xa8, 0xa4,
      0x8b, 0x01, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 0x53,
      0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01,
      0x5b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x57, 0x53, 0x53, 0x53, 0x53,
      0x12, 0x0b, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53,
      0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5b, 0x23, 0x00, 0xf2,
      0xa8, 0xa4, 0x8b, 0x01, 0x05, 0x00, 0x23, 0xf2, 0xa8, 0xa4, 0x8b, 0x01,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5e,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e,
      0x23, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0xf2,
      0xa8, 0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x53, 0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5b, 0x23, 0x00,
      0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x05, 0x00, 0x23, 0xf2, 0xa8, 0xa4, 0x8b,
      0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00, 0x5e,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x28, 0x5e, 0x23, 0x5e, 0x53, 0x53,
      0x12, 0x0b, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x68, 0x23, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59,
      0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x53, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x53, 0x00, 0x59, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x5b, 0x23, 0x00, 0xf2, 0xa8, 0xa4, 0x8b,
      0x01, 0x05, 0x5e, 0x23, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0xf2, 0xa8,
      0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53,
      0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x00, 0xf2, 0xa8, 0xa4, 0x8b,
      0x01, 0x6b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x74,
      0xcc, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
      0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x53, 0x00, 0x59, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x5b, 0x68, 0x23, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00,
      0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00,
      0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x23, 0x23, 0x8d, 0x8d,
      0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0xec, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x23,
      0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x05, 0x00, 0x23, 0xf2, 0xa8, 0xa4,
      0x8b, 0x01, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x5b, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x57, 0x53, 0x53, 0x53, 0x53, 0x12, 0x0b, 0x41, 0x12,
      0x41, 0x0b, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 0x53,
      0x53, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01,
      0x5b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0xf2, 0xa8, 0xa4, 0x8b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5b, 0x23, 0x00, 0xf2, 0xa8, 0xa4,
      0x8b, 0x01, 0x05, 0x00, 0x23, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5e, 0x5e, 0x5e,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x23, 0xf2,
      0xa8, 0xa4, 0x8b, 0x01, 0x00, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x5b, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53,
      0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x00, 0xf2, 0xa8, 0xa4,
      0x8b, 0xf2, 0xa8, 0xa4, 0x8b, 0x01, 0x05, 0x00, 0x23, 0xf2, 0xa8, 0xa4,
      0x8b, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00,
      0x5e, 0x5e, 0x5e, 0x5e, 0x5e, 0x0f, 0x01, 0x1e, 0x59, 0xda, 0x00, 0x1c,
      0x8d, 0x8d, 0x8d, 0xec, 0x08, 0xda, 0x59, 0x20, 0x59, 0xd6, 0xda, 0x59,
      0xdc, 0xda, 0x00, 0x00, 0x00, 0xa7, 0x29, 0x25, 0xa6, 0x23, 0xa7, 0xff,
      0xf9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x59, 0xda, 0xda,
      0xde, 0x20, 0x59, 0xd6, 0xda, 0x59, 0xdc, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4,
      0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4,
      0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0x59, 0xda, 0xda, 0xde, 0x20, 0x59, 0xf4,
      0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xfc, 0x00, 0x20, 0x00, 0xec, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00, 0xda,
      0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0xf4, 0xf4, 0xf4, 0x20, 0x59,
      0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x23, 0x23, 0x8d, 0x8d,
      0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xda, 0x59,
      0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00,
      0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x23, 0x23, 0x8d, 0x8d,
      0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0xec, 0xff,
      0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x23, 0x8d, 0x8d,
      0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x23, 0x23,
      0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00,
      0xec, 0x23, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59, 0xdc, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0xec, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59,
      0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0xec, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59, 0xdc, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59,
      0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00,
      0xec, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x23,
      0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x23,
      0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0xd6, 0xda, 0x59, 0xdc, 0xda, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0x23, 0x23, 0x8d, 0x8d, 0xec, 0x00, 0xda, 0x59, 0xdc, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x29, 0x00, 0x00, 0x00,
      0xec, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0x59, 0xdc, 0x00, 0x59, 0x26, 0x00, 0x00, 0xff, 0xdc, 0x00, 0x59,
      0x26, 0x00, 0x00, 0x00};
  constexpr std::size_t data_len_bytes = sizeof(data);
  constexpr std::size_t data_len = data_len_bytes / sizeof(char16_t);
  const auto validation1 = implementation.validate_utf16be_with_errors(
      (const char16_t *)data, data_len);
  ASSERT_EQUAL(validation1.error, simdutf::error_code::SURROGATE);
  ASSERT_EQUAL(validation1.count, 945);

  const bool validation2 =
      implementation.validate_utf16be((const char16_t *)data, data_len);
  ASSERT_EQUAL(validation1.error == simdutf::error_code::SUCCESS, validation2);

  const auto outlen = implementation.utf32_length_from_utf16be(
      (const char16_t *)data, data_len);
  std::vector<char32_t> output(outlen);
  const auto r = implementation.convert_utf16be_to_utf32_with_errors(
      (const char16_t *)data, data_len, output.data());
  ASSERT_EQUAL(r.error, simdutf::error_code::SURROGATE);
  ASSERT_EQUAL(r.count, 945);
}

TEST_LOOP(convert_2_UTF16_bytes) {
  // range for 1, 2 or 3 UTF-8 bytes
  simdutf::tests::helpers::RandomIntRanges random(
      {{0x0000, 0x007f}, {0x0080, 0x07ff}, {0x0800, 0xd7ff}, {0xe000, 0xffff}},
      seed);

  auto procedure = [&implementation](const char16_t *utf16, size_t size,
                                     char32_t *utf32) -> size_t {
    const simdutf::result res =
        implementation.convert_utf16be_to_utf32_with_errors(utf16, size, utf32);
    ASSERT_EQUAL(res.error, simdutf::error_code::SUCCESS);
    return res.count;
  };
  auto size_procedure = [&implementation](const char16_t *utf16,
                                          size_t size) -> size_t {
    return implementation.utf32_length_from_utf16be(utf16, size);
  };
  for (size_t size : input_size) {
    transcode_utf16_to_utf32_test_base test(BE, random, size);
    ASSERT_TRUE(test(procedure));
    ASSERT_TRUE(test.check_size(size_procedure));
  }
}

TEST_LOOP(convert_with_surrogates) {
  simdutf::tests::helpers::RandomIntRanges random(
      {{0x0800, 0xd800 - 1}, {0xe000, 0x10ffff}}, seed);

  auto procedure = [&implementation](const char16_t *utf16, size_t size,
                                     char32_t *utf32) -> size_t {
    const simdutf::result res =
        implementation.convert_utf16be_to_utf32_with_errors(utf16, size, utf32);
    ASSERT_EQUAL(res.error, simdutf::error_code::SUCCESS);
    return res.count;
  };
  auto size_procedure = [&implementation](const char16_t *utf16,
                                          size_t size) -> size_t {
    return implementation.utf32_length_from_utf16be(utf16, size);
  };
  for (size_t size : input_size) {
    transcode_utf16_to_utf32_test_base test(BE, random, size);
    ASSERT_TRUE(test(procedure));
    ASSERT_TRUE(test.check_size(size_procedure));
  }
}

TEST(convert_fails_if_there_is_sole_low_surrogate) {
  const size_t size = 64;
  transcode_utf16_to_utf32_test_base test(BE, []() { return '*'; }, size + 32);

  for (char16_t low_surrogate = 0xdc00; low_surrogate <= 0xdfff;
       low_surrogate++) {
    for (size_t i = 0; i < size; i++) {
      auto procedure = [&implementation, &i](const char16_t *utf16, size_t size,
                                             char32_t *utf32) -> size_t {
        const simdutf::result res =
            implementation.convert_utf16be_to_utf32_with_errors(utf16, size,
                                                                utf32);
        ASSERT_EQUAL(res.error, simdutf::error_code::SURROGATE);
        ASSERT_EQUAL(res.count, i);
        return 0;
      };
      const auto old = test.input_utf16[i];
      test.input_utf16[i] = to_utf16be(low_surrogate);
      ASSERT_TRUE(test(procedure));
      test.input_utf16[i] = old;
    }
  }
}

TEST(convert_fails_if_there_is_sole_high_surrogate) {
  const size_t size = 64;
  transcode_utf16_to_utf32_test_base test(BE, []() { return '*'; }, size + 32);

  for (char16_t high_surrogate = 0xdc00; high_surrogate <= 0xdfff;
       high_surrogate++) {
    for (size_t i = 0; i < size; i++) {
      auto procedure = [&implementation, &i](const char16_t *utf16, size_t size,
                                             char32_t *utf32) -> size_t {
        const simdutf::result res =
            implementation.convert_utf16be_to_utf32_with_errors(utf16, size,
                                                                utf32);
        ASSERT_EQUAL(res.error, simdutf::error_code::SURROGATE);
        ASSERT_EQUAL(res.count, i);
        return 0;
      };
      const auto old = test.input_utf16[i];
      test.input_utf16[i] = to_utf16be(high_surrogate);
      ASSERT_TRUE(test(procedure));
      test.input_utf16[i] = old;
    }
  }
}

TEST(
    convert_fails_if_there_is_low_surrogate_is_followed_by_another_low_surrogate) {
  const size_t size = 64;
  transcode_utf16_to_utf32_test_base test(BE, []() { return '*'; }, size + 32);

  for (char16_t low_surrogate = 0xdc00; low_surrogate <= 0xdfff;
       low_surrogate++) {
    for (size_t i = 0; i < size - 1; i++) {
      auto procedure = [&implementation, &i](const char16_t *utf16, size_t size,
                                             char32_t *utf32) -> size_t {
        const simdutf::result res =
            implementation.convert_utf16be_to_utf32_with_errors(utf16, size,
                                                                utf32);
        ASSERT_EQUAL(res.error, simdutf::error_code::SURROGATE);
        ASSERT_EQUAL(res.count, i);
        return 0;
      };
      const auto old0 = test.input_utf16[i + 0];
      const auto old1 = test.input_utf16[i + 1];
      test.input_utf16[i + 0] = to_utf16be(low_surrogate);
      test.input_utf16[i + 1] = to_utf16be(low_surrogate);
      ASSERT_TRUE(test(procedure));
      test.input_utf16[i + 0] = old0;
      test.input_utf16[i + 1] = old1;
    }
  }
}

TEST(convert_fails_if_there_is_surrogate_pair_is_followed_by_high_surrogate) {
  const size_t size = 64;
  transcode_utf16_to_utf32_test_base test(BE, []() { return '*'; }, size + 32);

  const char16_t low_surrogate = to_utf16be(0xd801);
  const char16_t high_surrogate = to_utf16be(0xdc02);
  for (size_t i = 0; i < size - 2; i++) {
    auto procedure = [&implementation, &i](const char16_t *utf16, size_t size,
                                           char32_t *utf32) -> size_t {
      const simdutf::result res =
          implementation.convert_utf16be_to_utf32_with_errors(utf16, size,
                                                              utf32);
      ASSERT_EQUAL(res.error, simdutf::error_code::SURROGATE);
      ASSERT_EQUAL(res.count, i + 2);
      return 0;
    };
    const auto old0 = test.input_utf16[i + 0];
    const auto old1 = test.input_utf16[i + 1];
    const auto old2 = test.input_utf16[i + 2];
    test.input_utf16[i + 0] = low_surrogate;
    test.input_utf16[i + 1] = high_surrogate;
    test.input_utf16[i + 2] = high_surrogate;
    ASSERT_TRUE(test(procedure));
    test.input_utf16[i + 0] = old0;
    test.input_utf16[i + 1] = old1;
    test.input_utf16[i + 2] = old2;
  }
}

#if SIMDUTF_CPLUSPLUS23

namespace {
template <auto input> constexpr auto size_be() {
  return simdutf::utf32_length_from_utf16le(input);
}
template <auto input> constexpr auto convert_be() {
  using namespace simdutf::tests::helpers;
  CTString<char32_t, size_be<input>()> tmp;
  const auto ret = simdutf::convert_utf16be_to_utf32_with_errors(input, tmp);
  if (ret.count != tmp.size()) {
    throw "unexpected write size";
  }
  return tmp;
}
} // namespace

TEST(compile_time_convert_utf16be_to_utf32_with_errors) {
  using namespace simdutf::tests::helpers;
  static_assert(convert_be<u"köttbulle"_utf16be>() == U"köttbulle"_utf32);
}

#endif

TEST_MAIN
